/*
    Copyright 2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <time.h>
#include "mydownlib.h"

#ifdef WIN32
    #include <winsock.h>

    #define close       closesocket
    #define TEMPOZ1
    #define TEMPOZ2     GetTickCount()
    #define ONESEC      1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/times.h>
    #include <sys/timeb.h>

    #define stristr     strcasestr
    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define TEMPOZ1     ftime(&timex)
    #define TEMPOZ2     ((timex.time * 1000) + timex.millitm)
    #define ONESEC      1
#endif



#define VER         "0.1"
#define DELAY       500
#define BUFFSZ      4096
#define MAXARGS     4               // modify it if you need more args in mydown_scanhead
// #define FSIZE       u_int       // actually only 32 bits are supported, no LFS
#define TEMPOZ(x)   TEMPOZ1; \
                    x = TEMPOZ2



void mydown_get_host(u_char *url, u_char **hostx, u_short *portx, u_char **urix, u_char **userx, u_char **passx, int verbose);
u_char *mydown_http_delimit(u_char *data);
u_char *mydown_uri2hex(u_char *uri);
u_char *mydown_hex2uri(u_char *uri);
void mydown_scanhead(u_char *data, int datalen, ...);
u_char *mydown_showhttp80(u_short port);
void mydown_showstatus(u_int fsize, u_int ret, u_int perc, u_int from, int difftime, int verbose);
u_char *mydown_base64_encode(u_char *data, int *length);
u_int mydown_resolve(char *host);



u_int mydown(u_char *myurl, u_char *filename, mydown_options *opt) {
    struct  stat    xstat;
    FILE    *fd          = NULL;
    u_int   from         = 0,
            tot          = 0,
            filesize     = -1;
    int     err,
            showhead     = 0,
            resume       = 0,
            onlyifdiff   = 0,
            verbose      = 0,
            download_now = 0;
    u_short port         = 0;
    u_char  *url         = NULL,
            *uri         = NULL,
            *host        = NULL,
            *user        = NULL,
            *pass        = NULL,
            *referer     = NULL,
            *useragent   = NULL,
            *cookie      = NULL,
            *more_http   = NULL,
            **filedata   = NULL,
            *p;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stderr, NULL);

    if(opt) {
        from         = opt->from;
        tot          = opt->tot;
        showhead     = opt->showhead;
        resume       = opt->resume;
        onlyifdiff   = opt->onlyifdiff;
        user         = opt->user;
        pass         = opt->pass;
        referer      = opt->referer;
        useragent    = opt->useragent;
        cookie       = opt->cookie;
        more_http    = opt->more_http;
        verbose      = opt->verbose;
        filedata     = opt->filedata;
        download_now = opt->download_now;

        if(user) {
            if(verbose > 0) fprintf(stderr,
                "  user   %s\n"
                "  pass   %s\n",
                user,
                pass);
        }
    }

    url = strdup(myurl);
    if(!url) return(-1);
    mydown_get_host(url, &host, &port, &uri, &user, &pass, verbose);

    if(filename) {
        filename = strdup(filename);    // required for easy free later
    }

    if(!download_now) {                 // request info
        if(verbose > 0) fprintf(stderr, "  check file and size\n");
        filesize = mydown_http2file(
            0,
            host,
            port,
            user,
            pass,
            referer,
            useragent,
            cookie,
            more_http,
            verbose,
            uri,
            NULL,
            &filename,
            showhead,
            from,
            tot,
            NULL,
            NULL);
        if(filesize == -1) goto quit;

        if(showhead) {
            filesize = 0;
            goto quit;
        }

        if(!filedata) {
            if(!filename || !filename[0]) {
                filename = mydown_hex2uri(uri);
                if(filename && filename[0]) {
                    for(
                      p = filename + strlen(filename) - 1;
                      (p >= filename) && (*p != '/') && (*p != '\\') && (*p != ':') && (*p != '&') && (*p != '?') && (*p != '=');
                      p--);
                    filename = p + 1;
                } else {
                    if(verbose >= 0) fprintf(stderr, "\nError: no filename retrieved, you must specify an output filename\n\n");
                    filesize = -1;
                    goto quit;
                }
            }
        }
    }

    if(filename) {
        if(!strcmp(filename, "-")) {
            fd = stdout;
            if(verbose >= 0) fprintf(stderr, "  file   %s\n", "stdout");
        } else {
            if(verbose >= 0) fprintf(stderr, "  file   %s\n", filename);
            err = stat(filename, &xstat);
            if(onlyifdiff && !err && (xstat.st_size == filesize)) {
                if(verbose >= 0) fprintf(stderr, "  the remote file has the same size of the local one, skip\n");
                filesize = -1;
                goto quit;
            }
            if((err < 0) || !resume) {      // file doesn't exist and must not resume
                fd = fopen(filename, "wb");
            } else {
                fd = fopen(filename, "ab");
                from = xstat.st_size;
                if(verbose > 0) fprintf(stderr, "  resume %u\n", from);
            }
            if(!fd) {
                filesize = -1;
                goto quit;
            }
        }
    }

    filename = NULL;
    if(verbose > 0) fprintf(stderr, "  start download\n");

    filesize = mydown_http2file(
        1,
        host,
        port,
        user,
        pass,
        referer,
        useragent,
        cookie,
        more_http,
        verbose,
        uri,
        fd,
        &filename,
        0,
        from,
        tot,
        NULL,
        filedata ? filedata : NULL);

quit:
    if(fd)       fclose(fd);
    if(url)      free(url);
    if(uri)      free(uri);
    if(filename) free(filename);
    return(filesize);
}



void mydown_get_host(u_char *url, u_char **hostx, u_short *portx, u_char **urix, u_char **userx, u_char **passx, int verbose) {
    u_short port  = 80;
    u_char  *host = NULL,
            *uri  = NULL,
            *user = NULL,
            *pass = NULL,
            *p;

    host = url;

    p = strstr(host, "://");    // handle http://
    if(!p) p = strstr(host, ":\\\\");
    if(p) {
        for(p += 3; *p; p++) {  // in case of http:////
            if((*p != '/') && (*p != '\\')) break;
        }
        host = p;
    }

    for(p = host; *p; p++) {    // search the uri
        if((*p == '/') || (*p == '\\')) {
            uri = p;
            break;
        }
    }
    if(uri) {
        *uri++ = 0;
        uri = mydown_uri2hex(uri);
    }
    if(!uri) uri = strdup("");  // in case mydown_uri2hex fails

    p = strchr(host, '@');
    if(p) {
        *p = 0;

        user = host;

        pass = strchr(host, ':');
        if(pass) {
            *pass++ = 0;
        } else {
            pass = "";
        }

        host = p + 1;
    }

    p = strchr(host, ':');
    if(p) {
        *p = 0;
        port = atoi(p + 1);
    }

    if(verbose >= 0) fprintf(stderr, "  %s\n", url);
    if(verbose > 0) fprintf(stderr,
        "  host   %s : %hu\n"
        "  uri    %s\n",
        host, port,
        uri);

    if(user) {
        if(verbose > 0) fprintf(stderr,
            "  user   %s\n"
            "  pass   %s\n",
            user,
            pass);
    }

    *hostx = host;
    *portx = port;
    *urix  = uri;
    *userx = user;
    *passx = pass;
}



u_char *mydown_http_delimit(u_char *data) {
    if(!data || !*data) return(NULL);
    while(*data && (*data != '\r') && (*data != '\n')) data++;
    *data = 0;
    for(data++; *data && ((*data == '\r') || (*data == '\n')); data++);
    return(data);
}



u_char *mydown_uri2hex(u_char *uri) {
    static const u_char hex[16] = "0123456789abcdef";
    u_char  *ret,
            *p,
            c;

    ret = malloc((strlen(uri) * 3) + 1);
    if(!ret) return(NULL);

    for(p = ret; *uri; uri++) {
        c = *uri;
        if((c >= 0x2f) && (c < 0x7f)) {
            *p++ = c;
        } else {
            *p++ = '%';
            *p++ = hex[c >> 4];
            *p++ = hex[c & 15];
        }
    }
    *p = 0;

    return(ret);
}



u_char *mydown_hex2uri(u_char *uri) {
    u_char  *ret,
            *p;

    ret = strdup(uri);
    if(!ret) return(NULL);

    for(p = ret; *uri; uri++, p++) {
        if(*uri == '%') {
            sscanf(uri + 1, "%02hhx", p);
            uri += 2;
        } else {
            *p = *uri;
        }
    }
    *p = 0;

    return(ret);
}



void mydown_scanhead(u_char *data, int datalen, ...) {
    va_list ap;
    int     i,
            vals;
    u_char  *par[MAXARGS],
            **val[MAXARGS],
            *l,
            *p,
            *limit;

    va_start(ap, datalen);
    for(i = 0; i < MAXARGS; i++) {
        par[i] = va_arg(ap, u_char *);
        if(!par[i]) break;
        val[i] = va_arg(ap, u_char **);
        if(!val[i]) break;
        *val[i] = NULL;
    }
    vals = i;
    va_end(ap);

    for(limit = data + datalen; (l = mydown_http_delimit(data)); data = l) {
        if(l > limit) break;
        p = strchr(data, ':');
        if(!p) continue;
        *p++ = 0;
        for(i = 0; i < vals; i++) {
            if(stricmp(data, par[i])) continue;
            while(*p && ((*p == ' ') || (*p == '\t'))) p++;
            *val[i] = p;
            break;
        }
    }
}



u_int mydown_http2file(int download, u_char *host, u_short port, u_char *user, u_char *pass, u_char *referer, u_char *useragent, u_char *cookie, u_char  *more_http, int verbose, u_char *getstr, FILE *fd, u_char **filename, int showhead, u_int from, u_int tot, u_int *filesize, u_char **filedata) {
#ifndef WIN32
    struct  timeb   timex;
#endif
    struct  sockaddr_in peer;
    time_t  oldtime      = 0,
            newtime;
    u_int   ret          = 0,
            fsize        = 0,
            perc         = 0;
    int     sd           = 0,
            t,
            len,
            code         = 0,
            b64len,
            filedatasz   = 0;
    u_char  *buff        = NULL,
            *query       = NULL,
            *data        = NULL,
            *p           = NULL,
            *s           = NULL,
            *userpass    = NULL,
            *b64         = NULL,
            *contlen     = NULL,
            *contdisp    = NULL,
            *icyname     = NULL,
            *location    = NULL,
            *filedatatmp = NULL;

#define GOTOQUIT    { ret = -1; goto quit; }

    peer.sin_addr.s_addr = mydown_resolve(host);
    if(!peer.sin_addr.s_addr) GOTOQUIT;
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) GOTOQUIT;

    if(verbose > 0) fprintf(stderr, "  connect...");
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer)) < 0) {
        fprintf(stderr, "\nError: connection refused\n");
        GOTOQUIT;
    }
    if(verbose > 0) fprintf(stderr, "done\n");

    if(user && pass) {
        userpass = malloc(strlen(user) + 1 + strlen(pass) + 1);
        if(!userpass) GOTOQUIT;
        b64len = sprintf(userpass, "%s:%s", user, pass);
        b64 = mydown_base64_encode(userpass, &b64len);
    }

    len =
        200                                         +   // my format strings
        strlen(getstr)                              +
        strlen(host)                                +
        5                                           +
        ((from || tot)   ? 10 + 10           : 0)   +
        (b64             ? strlen(b64)       : 0)   +
        (referer         ? strlen(referer)   : 0)   +
        (useragent       ? strlen(useragent) : 0)   +
        (cookie          ? strlen(cookie)    : 0)   +
        (more_http       ? strlen(more_http) : 0)   +
        1;

    query = malloc(len);
    if(!query) GOTOQUIT;

    len = sprintf(
        query,
        "GET /%s HTTP/1.0\r\n"
        "Host: %s%s\r\n"
        "Connection: close\r\n",
        getstr,
        host, mydown_showhttp80(port));

    if(from || tot) {
        len += sprintf(query + len, "Range: bytes=");
        if(from) len += sprintf(query + len, "%u", from);
        len += sprintf(query + len, "-");
        if(tot)  len += sprintf(query + len, "%u", tot + from);
        len += sprintf(query + len, "\r\n");
    }
    if(b64) {
        len += sprintf(
            query + len,
            "Authorization: Basic %s\r\n",
            b64);
    }
    if(referer) {
        len += sprintf(
            query + len,
            "Referer: %s\r\n",
            referer);
    }
    if(useragent) {
        len += sprintf(
            query + len,
            "User-Agent: %s\r\n",
            useragent);
    }
    if(cookie) {
        len += sprintf(
            query + len,
            "Cookie: %s\r\n",
            cookie);
    }
    if(more_http) {
        len += sprintf(
            query + len,
            "%s",
            more_http);
        if(query[len - 1] != '\n') len += sprintf(query + len, "\r\n");
    }
    len += sprintf(query + len, "\r\n");

    send(sd, query, len, 0);
    free(query);

    buff = malloc(BUFFSZ + 1);
    if(!buff) GOTOQUIT;

    data = p = buff;
    len  = BUFFSZ;
    while((t = recv(sd, data, len, 0)) > 0) {
        data += t;
        len  -= t;
        *data = 0;

        p = strstr(buff, "\r\n\r\n");
        if(p) {
            p += 4;
        } else {
            p = strstr(buff, "\n\n");
            if(p) p += 2;
        }

        if(p) {
            *(p - 1) = 0;

            if(showhead) {
                if(verbose > 0) fprintf(stderr, "\n%s", buff);
                goto quit;
            }

            s = strchr(buff, ' ');
            if(s) {
                code = atoi(s + 1);

                if((code / 100) == 3) {
                    mydown_scanhead(buff, p - buff,
                        "location",     &location,
                        NULL,           NULL);
                    if(!location) {
                        fprintf(stderr, "\nError: remote file is temporary unavailable (%d)\n", code);
                        GOTOQUIT;
                    }
                    if(verbose > 0) fprintf(stderr, "\n- redirect: %s\n", location);
                    mydown_get_host(location, &host, &port, &getstr, &user, &pass, verbose);
                    ret = mydown_http2file(download, host, port, user, pass, referer, useragent, cookie, more_http, verbose, getstr, fd, filename, showhead, from, tot, filesize, filedata);
                    goto quit;
                }

                    // if((code != 200) && (code != 206)) {
                if((code / 100) != 2) {
                    fprintf(stderr, "\nError: remote file is temporary unavailable (%d)\n", code);
                    GOTOQUIT;
                }
            }

            mydown_scanhead(buff, p - buff,
                "content-length",       &contlen,
                "content-disposition",  &contdisp,
                "icy-name",             &icyname,
                NULL,                   NULL);

            if(contlen) {
                s = strchr(contlen, '/');
                if(s) contlen = s + 1;
                sscanf(contlen, "%u", &fsize);
                perc = (from + fsize) / 100;
                if(!perc) perc++;
            }

            if(!contdisp && icyname) contdisp = icyname;
            if(filename && !*filename && contdisp) {
                s = (u_char *)stristr(contdisp, "filename=");
                if(!s) s = (u_char *)stristr(contdisp, "file=");
                if(s) {
                    s = strchr(s, '=') + 1;
                } else {
                    s = contdisp;
                }
                while(*s && ((*s == '\"') || (*s == ' ') || (*s == '\t'))) s++;
                *filename = strdup(s);
                for(s = *filename; *s; s++) {
                    if((*s == '\\') || (*s == '/') || (*s == ';') || (*s == ':') || (*s == '\"') || (*s == '&') || (*s == '?')) break;
                }
                for(s--; (s >= *filename) && *s && ((*s == ' ') || (*s == '\t')); s--);
                *(s + 1) = 0;
            }

            if(!download) {
                if(verbose > 0) fprintf(stderr, "  total filesize: %u", fsize);
                if(filesize) *filesize = fsize;
                ret = fsize;    // = 0???
                goto quit;
            }

            break;
        }
    }

    if(!p) p = buff;
    len = data - p;
    memmove(buff, p, len);

    if(verbose > 0) fprintf(stderr, "\n");
    if(fsize) if(verbose > 0) fprintf(stderr, "    ");
    if(verbose > 0) fprintf(stderr, " | downloaded | bytes/second\n");
    if(fsize) if(verbose > 0) fprintf(stderr, "----");
    if(verbose > 0) fprintf(stderr, "-/------------/-------------\n");

    TEMPOZ(oldtime);
    oldtime -= DELAY;
    ret = from;

    if(filedata) {
        filedatasz  = fsize;
        filedatatmp = malloc(filedatasz);
        if(!filedatatmp) GOTOQUIT;
    }

    do {
        ret += len;

        if(fd) {
            if(!fwrite(buff, len, 1, fd)) {
                fprintf(stderr, "\nError: I/O error. Probably your disk is full or the file is write protected\n");
                GOTOQUIT;
            }
            fflush(fd);
        }
        if(filedata) {
            if(filedatasz < ret) {
                filedatasz  = ret;
                filedatatmp = realloc(filedatatmp, filedatasz);
                if(!filedatatmp) GOTOQUIT;
            }
            memcpy(filedatatmp + filedatasz - len, buff, len);
        }

        TEMPOZ(newtime);
        if((newtime - oldtime) >= DELAY) {
            mydown_showstatus(fsize, ret, perc, from, (int)(newtime - oldtime), verbose);
            oldtime = newtime;
            from = ret;
        }
    } while((len = recv(sd, buff, BUFFSZ, 0)) > 0);

    TEMPOZ(newtime);
    mydown_showstatus(fsize, ret, perc, from, (int)(newtime - oldtime), verbose);

    if(filedata) {
        *filedata = filedatatmp;
    }

quit:
    if(userpass) free(userpass);
    if(b64)      free(b64);
    if(buff)     free(buff);
    if(sd)       close(sd);
    if(verbose > 0) fprintf(stderr, "\n");
    return(ret);
}



u_char *mydown_showhttp80(u_short port) {
    static u_char   mini[6];

    *mini = 0;
    if(port != 80) sprintf(mini, ":%hu", port);
    return(mini);
}



void mydown_showstatus(u_int fsize, u_int ret, u_int perc, u_int from, int difftime, int verbose) {
    if(fsize) if(verbose >= 0) fprintf(stderr, "%3u%%", ret / perc);
    if(verbose >= 0) fprintf(stderr, "   %10u", ret);
    ret -= from;
    if(ret > 0) if(verbose >= 0) fprintf(stderr, "   %-10u\r", (ret * 1000) / difftime);
    if(verbose >= 0) fprintf(stderr, "\r");
}



u_char *mydown_base64_encode(u_char *data, int *length) {
    int             r64len,
                    len = *length;
    u_char          *p64;
    static u_char   *r64;
    static const char   enctab[64] = {
        'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
        'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
        'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
        'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'
    };

    r64len = ((len / 3) << 2) + 5;
    r64 = malloc(r64len);
    if(!r64) return(NULL);
    p64 = r64;

    do {
        *p64++ = enctab[(*data >> 2) & 63];
        *p64++ = enctab[(((*data & 3) << 4) | ((*(data + 1) >> 4) & 15)) & 63];
        data++;
        *p64++ = enctab[(((*data & 15) << 2) | ((*(data + 1) >> 6) & 3)) & 63];
        data++;
        *p64++ = enctab[*data & 63];
        data++;

        len -= 3;
    } while(len > 0);

    for(; len < 0; len++) *(p64 + len) = '=';
    *p64 = 0;

    *length = p64 - r64;
    return(r64);
}



u_int mydown_resolve(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == htonl(INADDR_NONE)) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolve hostname (%s)\n\n", host);
            return(0);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}


