/*
    Copyright 2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#define QUAKE3_QUERY    "\xff\xff\xff\xff" "getstatus xxx\n"
#define MOH_QUERY       "\xff\xff\xff\xff" "\x02" "getstatus xxx\n"
#define QUAKE2_QUERY    "\xff\xff\xff\xff" "status"
#define HALFLIFE_QUERY  "\xff\xff\xff\xff" "infostring\n\0"
#define DPLAY8_QUERY    "\x00\x02"          /* info/join */     \
                        "\xff\xff"          /* ID tracker */    \
                        "\x02"              /* 01 = need GUID, 02 = no GUID */
#define DOOM3_QUERY     "\xff\xff" "getInfo\0" "\0\0\0\0"
#define ASE_QUERY       "s"
#define GS1_QUERY       "\\status\\"
#define GS2_QUERY       "\xfe\xfd\x00" "\x00\x00\x00\x00" "\xff\x00\x00" "\x00"
#define SOURCE_QUERY    "\xff\xff\xff\xff" "T"
#define TRIBES2_QUERY   "\x12\x02\x01\x02\x03\x04"



/*
stdout for any information
fdout only for the megaquery scanning
*/



void show_unicode(u_char *data, u_int size) {
    u_char  *limit = data + size;

    while(*data && (data <= limit)) {
        fputc(*data, stdout);
        data += 2;
    }
    fputc('\n', stdout);
}



int handle_query_par(u_char *data, int skip) {
    u_char  *p;

    p = strchr(data, '_');
    if(p) data = p + 1;     /* "sv_" and "si_" */

#define K(x)  !stricmp(data, x)
    if(!(skip & 2) && (
        K("hostname")   || K("name"))) {
        return(1);

    } else if(!(skip & 4) && (
        K("mapname")    || K("map"))) {
        return(2);

    } else if(!(skip & 8) && (
        K("gametype")   || K("gametypeLocal"))) {
        return(3);

    } else if(!(skip & 16) && (
        K("numplayers") || K("clients")  || K("privateClients"))) {
        return(4);

    } else if(!(skip & 32) && (
        K("maxplayers") || K("maxclients"))) {
        return(5);

    } else if(!(skip & 64) && (
        K("gamever")    || K("version")  || K("protocol"))) {
        return(6);

    } else if(!(skip & 128) && (
        K("password")   || K("needpass") || K("usepass")  || K("passworden") || K("pwd") || K("pswrd")  || K("pw"))) {
        return(7);

    } else if(!(skip & 256) && (
        K("dedicated")  || K("type")     || K("ded")      || K("dedic")      || K("serverDedicated"))) {
        return(8);

    } else if(!(skip & 512) && (
        K("hostport"))) {
        return(9);

    } else if(!(skip & 1024) && (
        K("game")       || K("gamedir")  || K("modname")  || K("mods")       || K("mod"))) {
        return(10);

    } else if(!(skip & 2048) && (
        K("pb")         || K("punkbuster"))) {
        return(11);

    } else if(!(skip & 4096) && (
        K("gamemode"))) {
        return(12);

    } else if(!(skip & 8192) && (
        K("ranked")     || K("rated"))) {
        return(13);
    }
#undef K
    return(0);
}



int print_query_par(u_char *data) {
    printf("%28s ", data);
    return(0);
}



void handle_query_value(u_char *data, int ipbit, ipdata_t *ipdata) {
    if(!*data) return;
    switch(ipbit) {
        case  1: ipdata->name    = mychrdup(data);      break;
        case  2: ipdata->map     = mychrdup(data);      break;
        case  3: ipdata->type    = mychrdup(data);      break;
        case  4: ipdata->players = atoi(data);          break;
        case  5: ipdata->max     = atoi(data);          break;
        case  6: ipdata->ver     = mychrdup(data);      break;
        case  7: ipdata->pwd     = *data;               break;
        case  8: ipdata->ded     = *data;               break;
        case  9: ipdata->port    = htons(atoi(data));   break;
        case 10: ipdata->mod     = mychrdup(data);      break;
        case 11: ipdata->pb      = *data;               break;
        case 12: ipdata->mode    = mychrdup(data);      break;
        case 13: ipdata->rank    = *data;               break;
        default: break;
    }
}



void print_query_value(u_char *data, int ipbit, ipdata_t *ipdata) {
    printf("%s\n", data);
}



void generic_query(u_char *data, int len, generic_query_data *gqd) {
    ipdata_t    *ipdata;
    int     ipbit   = 0,
            skip    = 0,
            nt,
            datalen = 0,
            first   = 0;
    u_char  *p,
            *tmp,
            *limit;
    int     (*par)();       /* the usage of function pointers */
    void    (*value)();     /* avoids to use many if() */

    limit  = data + len - gqd->rear;
    *limit = 0;
    data   += gqd->front;
    nt     = gqd->nt;

    if(gqd->ipdata) {
        ipdata = &gqd->ipdata[gqd->pos];
        par    = handle_query_par;
        value  = handle_query_value;
    } else {
        ipdata = NULL;
        par    = print_query_par;
        value  = print_query_value;
    }

    if(gqd->scantype == QUERY_SCANTYPE_SINGLE) {    // \par\value\par\value\...
        datalen = strlen(gqd->data);
        first   = 1;        // lame compatibility fix
    }

    for(p = data; (data < limit) && p; data = p + 1, nt++) {
        p = strchr(data, gqd->chr);
        if(p) *p = 0;

                /* work-around for Quake games with players at end, only for -Q */
        if(gqd->ipdata) {
            if(nt & 1) {
                if(skip && (gqd->chr != '\n')) {
                    for(tmp = data; *tmp; tmp++) {
                        if(*tmp == '\n') {
                            *tmp = 0;
                            p = limit;
                            break;
                        }
                    }
                }
            } else {
                skip |= (1 << ipbit);
            }
        }

        if(gqd->scantype == QUERY_SCANTYPE_SINGLE) {
            if(first && !*data) {
                first = 0;
                continue;
            }
            data_cleaner(data, data, strlen(data)); // format
            datalen += sprintf(gqd->data + datalen, "%s\\", data);
            continue;
        }

        if(nt & 1) {
            value(data, ipbit, ipdata);
            ipbit = 0;
        } else {
            if(!*data || !strcmp(data, "queryid") || !strcmp(data, "final")) break;
            ipbit = par(data, skip);
            skip |= (1 << ipbit);
        }
    }
}



void source_query(u_char *data, int len, generic_query_data *gqd) {
    ipdata_t   *ipdata;

    data += 5;
    if(gqd->ipdata) {
        ipdata = &gqd->ipdata[gqd->pos];

        data += strlen(data) + 1;
        ipdata->name    = mychrdup(data);               data += strlen(data) + 1;
        ipdata->map     = mychrdup(data);               data += strlen(data) + 1;
        ipdata->mod     = mychrdup(data);               data += strlen(data) + 1;
        ipdata->type    = mychrdup(data);               data += strlen(data) + 1;
        ipdata->players = *data++;
        ipdata->max     = *data++;
        ipdata->ver     = malloc(6);
        sprintf(ipdata->ver, "%hu", *data++);
        ipdata->ded     = (*data++ == 'd') ? '1' : '0';
        data++;
        ipdata->pwd     = *data;

    } else {
        printf("%28s %s\n",        "Address",          data);   data += strlen(data) + 1;
        printf("%28s %s\n",        "Hostname",         data);   data += strlen(data) + 1;
        printf("%28s %s\n",        "Map",              data);   data += strlen(data) + 1;
        printf("%28s %s\n",        "Mod",              data);   data += strlen(data) + 1;
        printf("%28s %s\n",        "Description",      data);   data += strlen(data) + 1;
        printf("%28s %hhu/%hhu\n", "Players",          data[0], data[1]);  data += 2;
        printf("%28s %hhu\n",      "Version",          *data++);
        printf("%28s %c\n",        "Server type",      *data++);
        printf("%28s %c\n",        "Server OS",        *data++);
        printf("%28s %hhu\n",      "Password",         *data++);
        printf("%28s %hhu\n",      "Secure server",    *data);
    }
}



void tribes2_query(u_char *data, int len, generic_query_data *gqd) {
    ipdata_t    *ipdata;
    int     sz,
            tmp;

#define TRIBES2STR(x)   sz = *data++;   \
                        tmp = data[sz]; \
                        data[sz] = 0;   \
                        x;              \
                        data += sz;     \
                        *data = tmp;

    data += 6;
    if(gqd->ipdata) {
        ipdata = &gqd->ipdata[gqd->pos];

        TRIBES2STR((ipdata->mod  = mychrdup(data)));
        TRIBES2STR((ipdata->type = mychrdup(data)));
        TRIBES2STR((ipdata->map  = mychrdup(data)));
        data++;
        ipdata->players = *data++;
        ipdata->max     = *data++;

    } else {
        TRIBES2STR(printf("%28s %s\n", "Mod",       data));
        TRIBES2STR(printf("%28s %s\n", "Game type", data));
        TRIBES2STR(printf("%28s %s\n", "Map",       data));
        data++;
        printf("%28s %hhu/%hhu\n", "Players", data[0], data[1]);   data += 2;
        printf("%28s %hhu\n",      "Bots",    *data);              data++;
        printf("%28s %hu\n",       "CPU",     *(short *)data);     data += 2;
        TRIBES2STR(printf("%28s %s\n", "Info",      data));
    }
#undef TRIBES2STR
}



void dplay8info(u_char *data, int len, generic_query_data *gqd) {
    struct myguid {
        uint32_t    g1;
        uint16_t    g2;
        uint16_t    g3;
        uint8_t     g4;
        uint8_t     g5;
        uint8_t     g6;
        uint8_t     g7;
        uint8_t     g8;
        uint8_t     g9;
        uint8_t     g10;
        uint8_t     g11;
    };
    struct dplay8_t {
        uint32_t    pcklen;
        uint32_t    reservedlen;
        uint32_t    flags;
        uint32_t    session;
        uint32_t    max_players;
        uint32_t    players;
        uint32_t    fulldatalen;
        uint32_t    desclen;
        uint32_t    dunno1;
        uint32_t    dunno2;
        uint32_t    dunno3;
        uint32_t    dunno4;
        uint32_t    datalen;
        uint32_t    appreservedlen;
        struct      myguid  instance_guid;
        struct      myguid  application_guid;
    } *dplay8;

    if(len < (sizeof(struct dplay8_t) + 4)) {
        printf("\nError: the packet received seems invalid\n");
        return;
    }

    dplay8 = (struct dplay8_t *)(data + 4);

    if(dplay8->session) {
        fputs("\nSession options:     ", stdout);
        if(dplay8->session & 1)   fputs("client-server ", stdout);
        if(dplay8->session & 4)   fputs("migrate_host ", stdout);
        if(dplay8->session & 64)  fputs("no_DPN_server ", stdout);
        if(dplay8->session & 128) fputs("password ", stdout);
    }

    printf("\n"
        "Max players          %u\n"
        "Current players      %u\n",
        dplay8->max_players,
        dplay8->players);

    printf("\n"
        "Instance GUID        %08x-%04hx-%04hx-%02hhx%02hhx-%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx",
        dplay8->instance_guid.g1,
        dplay8->instance_guid.g2,
        dplay8->instance_guid.g3,
        dplay8->instance_guid.g4,
        dplay8->instance_guid.g5,
        dplay8->instance_guid.g6,
        dplay8->instance_guid.g7,
        dplay8->instance_guid.g8,
        dplay8->instance_guid.g9,
        dplay8->instance_guid.g10,
        dplay8->instance_guid.g11);

    printf("\n"
        "Application GUID     %08x-%04hx-%04hx-%02hhx%02hhx-%02hhx%02hhx%02hhx%02hhx%02hhx%02hhx\n",
        dplay8->application_guid.g1,
        dplay8->application_guid.g2,
        dplay8->application_guid.g3,
        dplay8->application_guid.g4,
        dplay8->application_guid.g5,
        dplay8->application_guid.g6,
        dplay8->application_guid.g7,
        dplay8->application_guid.g8,
        dplay8->application_guid.g9,
        dplay8->application_guid.g10,
        dplay8->application_guid.g11);

        // removed any security check here

    fputs("\nGame description/Session name:\n\n  ", stdout);
    show_unicode(
        data + 4 + dplay8->fulldatalen,
        len - dplay8->fulldatalen);

    if(dplay8->appreservedlen) {
        printf("\nHexdump of the APPLICATION RESERVED data (%u) sent by the server:\n\n",
            dplay8->appreservedlen);
        show_dump(
            data + 4 + dplay8->datalen,
            dplay8->appreservedlen,
            stdout);
    }

    if(dplay8->reservedlen) {
        printf("\nHexdump of the RESERVED data (%u) sent by the server:\n\n",
            dplay8->reservedlen);
        show_dump(
            data + 4 + dplay8->fulldatalen + dplay8->desclen,
            dplay8->reservedlen,
            stdout);
    }
}



void ase_query(u_char *data, u_int size, generic_query_data *gqd) {
    u_char  *limit;
    int     num = 0;

    if(memcmp(data, "EYE1", 4)) {
        fwrite(data + 1, size - 1, 1, stdout);
        fputc('\n', stdout);
        return;
    }
    limit = data + size;
    data += 4;

    while((data < limit) && (*data > 1)) {
        switch(num) {
            case 0: printf("\n%28s ", "Running game"); break;
            case 1: printf("\n%28s ", "Game port");    break;
            case 2: printf("\n%28s ", "Server name");  break;
            case 3: printf("\n%28s ", "Battle mode");  break;
            case 4: printf("\n%28s ", "Map name");     break;
            case 5: printf("\n%28s ", "Version");      break;
            case 6: printf("\n%28s ", "Password");     break;
            case 7: printf("\n%28s ", "Players");      break;
            case 8: printf("\n%28s ", "Max players");  break;
            default: {
                if(num & 1) fputc('\n', stdout);
                } break;
        }
        fwrite(data + 1, *data - 1, 1, stdout);
        data += *data;
        num++;
    }

    fputc('\n', stdout);
    num = limit - data;
    if(num > 1) {
        fputs("\nHex dump of the rest of the packet:\n", stdout);
        show_dump(data, num, stdout);
    }
}



u_char *switch_type_query(int type, int *querylen, generic_query_data *gqd, void *func, int info) {
    u_char  *query,
            *msg;

    if(gqd) memset(gqd, 0, sizeof(generic_query_data));

#define ASSIGN(x,y,n,c,f,r,z) {                         \
            msg   = x;                                  \
            query = y;                                  \
            if(querylen) *querylen  = sizeof(y) - 1;    \
            if(gqd) {                                   \
                gqd->nt    = n;                         \
                gqd->chr   = c;                         \
                gqd->front = f;                         \
                gqd->rear  = r;                         \
            }                                           \
            if(func) *(u_char **)func = (void *)z;      \
        } break;

    switch(type) {
        case  0: ASSIGN("Gs \\status\\",  GS1_QUERY,      1, '\\',  0, 0, generic_query);
        case  1: ASSIGN("Quake 3 engine", QUAKE3_QUERY,   1, '\\',  4, 0, generic_query);
        case  2: ASSIGN("Medal of Honor", MOH_QUERY,      1, '\\',  5, 0, generic_query);
        case  3: ASSIGN("Quake 2 engine", QUAKE2_QUERY,   1, '\\',  4, 0, generic_query);
        case  4: ASSIGN("Half-Life",      HALFLIFE_QUERY, 1, '\\', 23, 0, generic_query);
        case  5: ASSIGN("DirectPlay 8",   DPLAY8_QUERY,   0, '\0',  0, 0, dplay8info);
        case  6: ASSIGN("Doom 3 engine",  DOOM3_QUERY,    0, '\0', 23, 8, generic_query);
        case  7: ASSIGN("All-seeing-eye", ASE_QUERY,      0, '\0',  0, 0, ase_query);
        case  8: ASSIGN("Gamespy 2",      GS2_QUERY,      0, '\0',  5, 0, generic_query);
        case  9: ASSIGN("Source",         SOURCE_QUERY,   0, '\0',  0, 0, source_query);
        case 10: ASSIGN("Tribes 2",       TRIBES2_QUERY,  0, '\0',  0, 0, tribes2_query);
        default: return(NULL);
    }
#undef ASSIGN

    switch(info) {
        case 1: fprintf(stderr, "- query type: %s\n", msg); break;  /* details */
        case 2: printf(" %2d %-22s", type, msg);            break;  /* list */
        case 3: return(msg);                                break;  /* type */
    }
    return(query);
}



    /* this is the code called when you use the -i, -I or -d option */
void multi_query(int type, uint32_t ip, uint16_t port) {
    void    (*func)();
    generic_query_data  gqd;
    struct  sockaddr_in peer;
    int     sd,
            len,
            querylen;
    u_char  buff[4096],
            *query;

    query = switch_type_query(type, &querylen, &gqd, &func, 1);
    if(!query) return;

    peer.sin_addr.s_addr = ip;
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    fprintf(stderr, "- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    sendto(sd, query, querylen, 0, (struct sockaddr *)&peer, sizeof(peer));
    if(timeout(sd, TIMEOUT) < 0) {
        fputs("\n"
            "Error: Socket timeout, no reply received\n"
            "\n", stderr);
        exit(1);
    }
    len = recvfrom(sd, buff, sizeof(buff) - 1, 0, NULL, NULL);
//    if(len < 0) std_err();
    if(len > 0) {
        gqd.scantype = QUERY_SCANTYPE_GSWEB;
        gqd.data     = malloc(len + 1);
        gqd.data[0]  = 0;
        gqd.ipdata   = NULL;
        (*func)(buff, len, &gqd);
        free(gqd.data);
    }
    close(sd);
}



    /* receives all the replies from the servers */
MSTHREAD(multi_scan_reply, generic_query_data *gqd) {
    struct  sockaddr_in peer;
#ifndef WIN32
    struct  timeb   timex;
#endif
    int     sd,
            i,
            len,
            psz;
    u_char  buff[4096];

    sd  = gqd->sock;
    psz = sizeof(peer);

    while(!timeout(sd, gqd->maxping)) {
        len = recvfrom(sd, buff, sizeof(buff) - 1, 0, (struct sockaddr *)&peer, &psz);
        if(len < 0) continue;

        if(gqd->scantype == QUERY_SCANTYPE_SINGLE) {
            gqd->data = malloc(25 + len + 1);     // 15 IP + 1 + 5 port + 1 + len
            sprintf(gqd->data, "%s:%hu ", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));
            gqd->pos = 0;
            goto doit;
        }

        for(i = 0; gqd->ipdata[i].ip; i++) {
            if((gqd->ipdata[i].ip == peer.sin_addr.s_addr) && (gqd->ipdata[i].port == peer.sin_port)) {
                break;
            }
        }
        if(!gqd->ipdata[i].ip) {
            continue;
        }

        if(gqd->ipdata[i].sort == IPDATA_SORT_CLEAR) {
            TEMPOZ1;
            gqd->ipdata[i].ping = TEMPOZ2 - gqd->ipdata[i].ping;
            gqd->ipdata[i].sort = IPDATA_SORT_REPLY;
        }

        gqd->pos = i;

doit:
        (*gqd->func)(buff, len, gqd);

        if(gqd->scantype == QUERY_SCANTYPE_SINGLE) {
            if(sql) {
#ifdef SQL
                gssql(gqd->data);
#endif
            } else {
                fprintf(fdout, "%s\n", gqd->data);
            }
            free(gqd->data);
        }
    }

    return(0);
}



    /* this function is used by gsweb consider it legacy! */
int multi_scan(int type, void *ipbuff, int iplen, ipdata_t *gip, u_int ping) {
    ipport_t    *ipport;
    void    (*func)();
    generic_query_data  gqd;
    struct  sockaddr_in peer;
    u_int   servers;
    int     sd,
            querylen;
    u_char  *query;

#ifdef WIN32
    DWORD       tid,
                ret;
    HANDLE      th;
    #define MSTHREADF   for(;;) {                           \
                            GetExitCodeThread(th, &ret);    \
                            if(!ret) break;                 \
                            sleep(100);                     \
                        }
#else
    struct  timeb   timex;
    pthread_t   tid;
    int         th;
    #define MSTHREADF   pthread_join(tid, NULL);
#endif

    ipport = (ipport_t *)ipbuff;
    if(!ping) goto no_ping;

    query = switch_type_query(type, &querylen, &gqd, &func, 0);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

//    memset(gip, 0, (servers + 1) * sizeof(ipdata_t));   // useless?

    gqd.sock     = sd;
    gqd.maxping  = ping;
    gqd.ipdata   = gip;
    gqd.func     = func;
    gqd.scantype = QUERY_SCANTYPE_GSWEB;
    MSTHREADX(th, tid, multi_scan_reply, &gqd);

    peer.sin_family = AF_INET;

    for(servers = 0; iplen; servers++, ipport++, iplen--) {
        peer.sin_addr.s_addr = gip[servers].ip   = ipport->ip;
        peer.sin_port        = gip[servers].port = ipport->port;

        TEMPOZ1;
        gip[servers].ping = TEMPOZ2;

        sendto(sd, query, querylen, 0, (struct sockaddr *)&peer, sizeof(peer));
        usleep(SCANDELAY);
    }

    MSTHREADF;

    close(sd);
    return(servers);

no_ping:
    for(servers = 0; iplen; servers++, ipport++, iplen--) {
        gip[servers].ip   = ipport->ip;
        gip[servers].port = ipport->port;
        gip[servers].sort = IPDATA_SORT_REPLY;
    }
    return(servers);
}



    /* sends query packets to all the hosts we have received (ipbuff) */
int mega_query_scan(int type, void *ipbuff, int iplen, u_int ping) {
    ipport_t    *ipport;
    void    (*func)();
    generic_query_data  gqd;
    struct  sockaddr_in peer;
    u_int   servers;
    int     sd,
            querylen;
    u_char  *query;

#ifdef WIN32
    DWORD       tid,
                ret;
    HANDLE      th;
    #define MSTHREADF   for(;;) {                           \
                            GetExitCodeThread(th, &ret);    \
                            if(!ret) break;                 \
                            sleep(100);                     \
                        }
#else
    pthread_t   tid;
    int         th;
    #define MSTHREADF   pthread_join(tid, NULL);
#endif

    ipport = (ipport_t *)ipbuff;

    query = switch_type_query(type, &querylen, &gqd, &func, 0);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    gqd.sock     = sd;
    gqd.maxping  = ping;
    gqd.func     = func;
    gqd.scantype = QUERY_SCANTYPE_SINGLE;
    MSTHREADX(th, tid, multi_scan_reply, &gqd);

    peer.sin_family = AF_INET;

    for(servers = 0; iplen; servers++, ipport++, iplen--) {
        peer.sin_addr.s_addr = ipport->ip;
        peer.sin_port        = ipport->port;

        sendto(sd, query, querylen, 0, (struct sockaddr *)&peer, sizeof(peer));
        usleep(SCANDELAY);
    }

    MSTHREADF;

    close(sd);
    return(servers);
}


