/*

======
Gslist
======

INTRODUCTION
------------
Gslist is a game servers browser and heartbeats sender, it supports an
incredible amount of games and moreover a lot of options making it
simple and complete at the same time.
Then it is opensource and can be compiled on any system with small
changes (if needed).

All this software is based on the FREE and OFFICIAL PUBLIC data
available online just by the same Gamespy on their PUBLIC servers.


CONTRIBUTORS
------------
- Steve Hartland:  filters (cool) and FREEBSD compatibility
- Cimuca Cristian: bug reports and ideas
- Ludwig Nussel:   gslist user's directory (very useful in Unix), quiet
                   option and other tips
- Ben "LordNikon": web interface optimizations
- Kris Sum:        suggestions for the web interface
- KaZaMa:          many suggestions for the web interface
- all the others that I don't remember due to my limited memory 8-)


LICENSE
-------
    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <stdarg.h>
#include <stdint.h>

#ifdef WIN32
    #include <winsock.h>
    #include <windows.h>
    #include <direct.h>
    #include <io.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define usleep      sleep
    #define ftruncate   chsize
    #define in_addr_t   uint32_t
    #define ONESEC      1000
    #define SCANDELAY   3       // milliseconds, remember: modify not_WIN32 too
    #define TEMPOZ1
    #define TEMPOZ2     GetTickCount()
    #define APPDATA     "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders"
    #define PATH_SLASH  '\\'
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/times.h>
    #include <sys/timeb.h>
    #include <pthread.h>

    #define ONESEC      1
    #define SCANDELAY   3000    // remember * 1000: modify WIN32 too
    #define stristr     strcasestr
    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define TEMPOZ1     ftime(&timex)
    #define TEMPOZ2     ((timex.time * 1000) + timex.millitm)
    #define PATH_SLASH  '/'
#endif



#define VER             "0.8.2"
#define MS              "master.gamespy.com"
#define MSPORT          28900
#define HBPORT          27900
#define BUFFSZ          8192
#define GSHB1           "\\heartbeat\\%hu" \
                        "\\gamename\\%s"
#define GSHB2a          "\\validate\\%s" \
                        "\\final\\"
#define GSHB2b          "\\gamename\\%s" \
                        "\\final\\"
#define PCK             "\\gamename\\%s" \
                        "\\enctype\\%d" \
                        "\\validate\\%s" \
                        "\\final\\" \
                        "\\list\\cmp" \
                        "\\gamename\\%s" \
                        "%s%s"
#define HOST            "motd.gamespy.com"
#define ALUIGIHOST      "aluigi.org"
#define GETSZ           2048
#define GSUSERAGENT     "GameSpyHTTP/1.0"
#define TIMEOUT         2
#define HBMINUTES       300  // 300 seconds = 5 minutes
#define FILEOUT         "gslist-out.gsl"
#define GSLISTCFG       "gslist.cfg"
#define GSLISTTMP       "gslist.tmp"
#define FULLCFG         "full.cfg"
#define KNOWNCFG        "knownsvc.cfg"
#define GSHKEYSCFG      "gshkeys.txt"
#define DETECTCFG       "detection.cfg"
#define OLDGSINFO       "\\status\\"
#define MAXNAMESREQ     60
#define GSLISTSZ        80
#define FULLSZ          2048
#define DETECTSZ        300
#define KNOWNSZ         64
#define CFNAMEOFF       0
#define CNAMEOFF        54
#define CKEYOFF         73
#define CFNAMELEN       ((CNAMEOFF - CFNAMEOFF) - 1)
#define CNAMELEN        ((CKEYOFF - CNAMEOFF) - 1)
#define CFNAMEEND       (CFNAMEOFF + CFNAMELEN)
#define CNAMEEND        (CNAMEOFF + CNAMELEN)
#define GSTITLE         "\n" \
                        "Gslist "VER"\n" \
                        "by Luigi Auriemma\n" \
                        "e-mail: aluigi@autistici.org\n" \
                        "web:    aluigi.org\n" \
                        "\n"
#define GSLISTVER       "GSLISTVER: "
#define GSLISTHOME      "http://aluigi.org/papers.htm#gslist"
#define GSMAXPATH       256

#define DOITLATER_LIST  1
#define DOITLATER_UPD   2
#define DOITLATER_CFG   3
#define DOITLATER_CFG0  4

#define GSW_SORT_PING   0
#define GSW_SORT_NAME   1
#define GSW_SORT_MAP    2
#define GSW_SORT_TYPE   3
#define GSW_SORT_PLAYER 4
#define GSW_SORT_VER    5
#define GSW_SORT_MOD    6
#define GSW_SORT_DED    7
#define GSW_SORT_PWD    8
#define GSW_SORT_PB     9
#define GSW_SORT_RANK   10
#define GSW_SORT_MODE   11

#define ARGHELP         !argv[i] || !strcmp(argv[i], "?") || !stricmp(argv[i], "help")
#define ARGHELP1        !argv[i] || !argv[i + 1] || !strcmp(argv[i + 1], "?") || !stricmp(argv[i + 1], "help")
#define STRCPY(x,y)     mystrcpy(x, y, sizeof(x));
//#define WEBCPY(x,y)     web_strcpy(x, y, sizeof(x));

#ifdef WIN32
    #define MSTHREAD(NAME,ARG)  DWORD WINAPI NAME(ARG)
    #define MSTHREADX(TH,TID,THR,PAR)                                   \
        TH = CreateThread(NULL, 0, (void *)&THR, (void *)PAR, 0, &TID); \
        if(!TH) {                                                       \
            fprintf(stderr, "\nError: Unable to create thread\n\n");    \
            exit(1);                                                    \
        }
#else
    #define MSTHREAD(NAME,ARG)  void *NAME(ARG)
    #define MSTHREADX(TH,TID,THR,PAR)                                   \
        TH = pthread_create(&TID, NULL, (void *)&THR, (void *)PAR);     \
        if(TH) {                                                        \
            fprintf(stderr, "\nError: Unable to create thread\n\n");    \
            exit(1);                                                    \
        }
#endif



void prepare_query(int query, u_char *ip, u_char *port);
int gsfullup_aluigi(void);



#pragma pack(1)             // save tons of memory

typedef struct {
    uint32_t    ip;         // 0.0.0.0
    uint16_t    port;
    u_char      *name;      // string
    u_char      *map;       // string
    u_char      *type;      // string
    uint8_t     players;    // 0-255
    uint8_t     max;        // 0-255
    u_char      *ver;       // string
    u_char      *mod;       // string
    u_char      *mode;      // string
    uint8_t     ded;        // 0-255
    uint8_t     pwd;        // 0-255
    uint8_t     pb;         // 0-255
    uint8_t     rank;       // 0-255
    uint8_t     sort;       // 0 = clear, 1 = reply, 2 = sorted
    #define IPDATA_SORT_CLEAR   0
    #define IPDATA_SORT_REPLY   1
    #define IPDATA_SORT_SORTED  2
    uint16_t    ping;       // 0-65535
} ipdata_t;

typedef struct {
    uint8_t     nt;         // 0 or 1
    uint8_t     chr;        // the delimiter
    uint8_t     front;      // where the data begins?
    uint8_t     rear;       // where the data ends?
    uint16_t    maxping;
    int         sock;       // scan only
    int         pos;        // scan only
    void        (*func)();  // scan only
    uint8_t     scantype;   // scan only, 1 for gsweb and single queries
    #define QUERY_SCANTYPE_SINGLE   0
    #define QUERY_SCANTYPE_GSWEB    1
    u_char      *data;      // scan only, if 0
    ipdata_t    *ipdata;    // scan only, if 1
} generic_query_data;

typedef struct {
    u_char      *game;      /* gamename */
    u_char      *key;       /* gamekey */
    u_char      *full;      /* full game description */
    u_char      *path;      /* executable path */
    u_char      *filter;    /* filter */
    uint8_t     query;      /* type of query */
} gsw_data_t;

typedef struct {
    u_char      *game;      /* gamename */
    u_char      *pass;
    uint32_t    ip;
    uint16_t    port;
} gsw_fav_data_t;

typedef struct {
    u_char      *full;
    u_char      *game;
    u_char      *exe;
} gsw_scan_data_t;

struct {
    u_char      *game;
    u_char      *buff;
    int         len;
} gsw_refresh;

typedef struct {
    uint32_t    ip;
    uint16_t    port;
} ipport_t;

#pragma pack()



FILE    *fdout;                 // for the -o option
int     megaquery       = -1,
        sql             = 0,
        quiet           = 0,
        enctype         = 1;        // 0, 1 and 2
u_short msport          = MSPORT;
u_char  *ms             = MS,
        *msgamename     = "gamespy2\\gamever\\20603020",
        *msgamekey      = "d4kZca",
        gslist_path[GSMAXPATH + 1],
        *sql_host       = NULL,
        *sql_database   = NULL,
        *sql_username   = NULL,
        *sql_password   = NULL,
        *sql_query      = NULL,
        *sql_queryb     = NULL,
        *sql_queryl     = NULL;



#include "countries.h"          // list of countries
#include "gsmyfunc.h"           // mystrcpy, mywebcpy and so on
#include "gslegacy.h"           // mostly gsweb related
#include "gsshow.h"             // help, output, format and so on
#include "mydownlib.h"          // real file downloading
#include "gshttp.h"             // easy to use file downloading function
#include "gscfg.h"              // manual configuration builder (used essentially for me)
extern u_char *enctype1_decoder(u_char *, u_char *, int *); // for enctype1 decoding
extern u_char *enctype2_decoder(u_char *, u_char *, int *); // for enctype2 decoding
#include "gsmsalg.h"            // the old magician
#include "show_dump.h"          // hex dump of data
#ifdef SQL
    #include "gssql.h"          // all the SQL part
#endif
#include "multi_query.h"        // scanning and queries
#ifdef GSWEB
    #include "gswskin.h"        // internal index page
    #include "gsweb.h"          // web interface
#endif



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer,
                        peerl;
    ipport_t    *ipport,
                *ipbuffer;
    u_int   servers,
            msip;
    int     sd,
            err,
            len,
            i,
            execlen          = 0,
            hbmethod         = 0,
            psz,
            dynsz,
            itsok            = 0,
            on               = 1,
            iwannaloop       = 0,
            doitlater_type   = 0;
    u_short heartbeat_port   = 0;
    u_char  *buff            = NULL,
            *gamestr         = NULL,
            validate[43],
            secure[65],
            *sec             = NULL,
            outtype          = 0,
            *tmpexec         = NULL,
            *execstring      = NULL,
            *execstring_ip   = NULL,
            *execstring_port = NULL,
            *filter          = "",
            *execptr         = NULL,
            *doitlater_str   = NULL,
            *multigamename   = NULL,
            *multigamenamep  = NULL,
            *ipc             = NULL,
            *fname           = NULL;

#ifdef GSWEB
    u_short gswport          = 0;
    u_char  *gswip           = NULL,
            *gswopt          = NULL,
            *gswe            = NULL,
            *gswc            = NULL;
#endif

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdin,  NULL);
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fprintf(stderr, GSTITLE);

    fdout = stdout;
    *gslist_path = 0;

#ifdef WINTRAY
    gswip   = "127.0.0.1";
    gswport = 80;
    gsweb(resolv(gswip), gswport);
    return(0);
#endif

    if(argc < 2) {
        show_help();
        exit(1);
    }

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            fprintf(stderr, "\n"
                "Error: recheck your options (%s is not valid)\n"
                "\n", argv[i]);
            exit(1);
        }

        switch(argv[i][1]) {
            case '-':
            case '/':
            case '?':
            case 'h': {
                show_help();
                return(0);
                } break;
            case 'n':
            case 'N': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must select a gamename\n"
                        "Use -l for the full list or -s for searching a specific game\n"
                        "\n");
                    exit(1);
                }
                gamestr = argv[i];
                } break;
            case 'l': {
                doitlater_type = DOITLATER_LIST;
                } break;
            case 's': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify a text pattern to search in the game database\n"
                        "\n");
                    exit(1);
                }
                doitlater_type = DOITLATER_LIST;
                doitlater_str  = argv[i];
                } break;
            case 'u':
            case 'U': {
                doitlater_type = DOITLATER_UPD;
                } break;
            case 'i': {
                i++;
                prepare_query(0, argv[i], argv[i + 1]);
                return(0);
                } break;
            case 'I': {
                i++;
                prepare_query(8, argv[i], argv[i + 1]);
                return(0);
                } break;
            case 'd': {
                i++;
                if(ARGHELP1) {
                    printf("- list of supported game queries and their number:\n");
                    i = 0;
                    while(switch_type_query(i, NULL, NULL, NULL, 2)) {
                        if(!(++i % 3)) fputc('\n', stdout);
                    }
                    fputc('\n', stdout);
                    return(0);
                }
                prepare_query(atoi(argv[i]), argv[i + 1], argv[i + 2]);
                return(0);
                } break;
            case 'f': {
                i++;
                if(ARGHELP) {
                    show_filter_help();
                    return(0);
                }
                filter = argv[i];
                } break;
            case 'r': {
                i++;
                execstring = argv[i];
                execlen = strlen(execstring) + 23;

                tmpexec = malloc(execlen);
                if(!tmpexec) std_err();

                execstring_ip = strstr(execstring, "#IP");
                execstring_port = strstr(execstring, "#PORT");
                if(execstring_ip) *execstring_ip = 0;
                if(execstring_port) *execstring_port = 0;

                execlen = strlen(execstring);
                memcpy(tmpexec, execstring, execlen);
                } break;
            case 'o': {
                i++;
                if(ARGHELP) {
                    show_output_help();
                    return(0);
                }
                outtype = atoi(argv[i]);
                if(outtype > 6) outtype = 0;
                if(!outtype) {
                    fdout = fopen(argv[i], "wb");
                    if(!fdout) std_err();
                }
                } break;
#ifdef GSWEB
            case 'w': {
                i++;
                if(!argv[i] || !argv[i + 1]) {
                    fprintf(stderr, "\n"
                        "Error: you need to specify the local IP and port to bind\n"
                        "\n");
                    exit(1);
                }
                gswip   = argv[i];
                gswport = atoi(argv[++i]);
                } break;
            case 'W': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you need to specify the options for gsweb separated by comma:\n"
                        "       refresh=0   disable the refresh buffer/button\n"
                        "\n");
                    exit(1);
                }
                gswopt = argv[i];
                do {
                    gswe = strchr(gswopt, '=');
                    gswc = strchr(gswopt, ',');
                    if(gswc) *gswc = 0;
                    if(gswe) {
                        *gswe++ = 0;
                    } else {
                        gswe = "1";
                    }
                    if(!stricmp(gswopt, "refresh")) gsw_allow_refresh = (atoi(gswe) ? 1 : 0);
                    if(!stricmp(gswopt, "admin"))   gsw_admin_IP      = gswe;
                    gswopt = gswc + 1;
                } while(gswc);
                } break;
#endif
            case 'q': {
                quiet = 1;
                } break;
            case 'x': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify the master server and optionally its port\n"
                        "\n");
                    exit(1);
                }
                ms = strchr(argv[i], ':');
                if(ms) {
                    msport = atoi(ms + 1);
                    *ms = 0;
                }
                ms = argv[i];
                } break;
            case 'b': {
                i++;
                msport = HBPORT;
                heartbeat_port = atoi(argv[i]);
                hbmethod = 1;
                } break;
            case 'B': {
                i++;
                msport = HBPORT;
                heartbeat_port = atoi(argv[i]);
                hbmethod = 2;
                } break;
            case 'L': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify the amount of seconds for the loop\n"
                        "\n");
                    exit(1);
                }
                iwannaloop = atoi(argv[i]);
                } break;
            case 't': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must select an enctype number\n"
                        "\n");
                    exit(1);
                }
                enctype = atoi(argv[i]);
                } break;
            case 'c': {
                show_countries();
                return(0);
                } break;
            case 'y': {
                i++;
                if(!argv[i] || !argv[i + 1]) {
                    fprintf(stderr, "\n"
                        "Error: you must chose a gamename and a game key\n"
                        "\n");
                    exit(1);
                }
                gamestr = argv[i];
                i++;
                // retro-compatibility only
                } break;
            case 'Y': {
                i++;
                if(!argv[i] || !argv[i + 1]) {
                    fprintf(stderr, "\n"
                        "Error: you must chose a gamename and a gamekey\n"
                        "\n");
                    exit(1);
                }
                msgamename = argv[i];
                i++;
                msgamekey = argv[i];
                if(strlen(msgamekey) > (((sizeof(validate) - 1) * 3) / 4)) {
                    fprintf(stderr, "\n"
                        "Error: the gamekey you have specified is too long\n"
                        "\n");
                    exit(1);
                }
                } break;
            case 'p': {
                i++;
                len = STRCPY(gslist_path, argv[i]);
                if(gslist_path[len - 1] != PATH_SLASH) {
                    gslist_path[len]     = PATH_SLASH;
                    gslist_path[len + 1] = 0;
                }
                } break;
            case 'm': {
                doitlater_type = DOITLATER_CFG;
                } break;
            case 'M': {
                doitlater_type = DOITLATER_CFG0;
                } break;
            case 'Q': {
                i++;
                if(ARGHELP) {
                    printf("- list of supported game queries and their number:\n");
                    i = 0;
                    while(switch_type_query(i, NULL, NULL, NULL, 2)) {
                        if(!(++i % 3)) fputc('\n', stdout);
                    }
                    fputc('\n', stdout);
                    return(0);
                }
                megaquery = atoi(argv[i]);
                } break;
#ifdef SQL
            case 'S': {
                i++;
                if(ARGHELP) {
                    show_sql();
                    exit(1);
                }
                if(!argv[i] || !argv[i + 1] || !argv[i + 2] || !argv[i + 3] || !argv[i + 4] || !argv[i + 5] || !argv[i + 6]) {
                    fprintf(stderr, "\n"
                        "Error: some parameters are missed, use -S to see all the required arguments\n"
                        "\n");
                    exit(1);
                }
                sql = 1;
        #define AUTONULL(x) x = argv[i++];      \
                            if(!x[0]) x = NULL;
                AUTONULL(sql_host);
                if(!sql_host) sql_host = "127.0.0.1";
                sql_database = argv[i++];
                AUTONULL(sql_username);
                AUTONULL(sql_password);
                sql_query    = argv[i++];
                AUTONULL(sql_queryb);
                AUTONULL(sql_queryl);
        #undef AUTONULL
                } break;
#endif
            case 'e': {
                show_examples();
                return(0);
                } break;
            case 'v': {
                fdout = gslfopen(GSLISTCFG, "rb");
                if(fdout) fclose(fdout);
                printf(
                    "Gslist home      "GSLISTHOME"\n"
                    "Database folder  %s\n"
                    "\n", gslist_path);
                return(0);
                } break;
            default: {
                fprintf(stderr, "\n"
                    "Error: wrong argument (%s)\n"
                    "\n", argv[i]);
                exit(1);
                } break;
        }
    }

#ifdef GSWEB
    if(gswip) {
        if(!gswport) gswport = 80;
        gsweb(resolv(gswip), gswport);
        return(0);
    }
#endif

    if(doitlater_type) {
        switch(doitlater_type) {
            case DOITLATER_LIST:    show_list(doitlater_str);   break;
            case DOITLATER_UPD:     gsfullup_aluigi();          break;
            case DOITLATER_CFG:     make_gslistcfg(0);          break;
            case DOITLATER_CFG0:    make_gslistcfg(1);          break;
            default: break;
        }
        return(0);
    }

    if(!gamestr) {
        fprintf(stderr, "\n"
            "Error: The game is not available or has not been specified.\n"
            "       Use -n to specify a gamename\n"
            "\n");
        exit(1);
    }

    if(!quiet) {
        fprintf(fdout,
            "Gamename:    %s\n"
            "Enctype:     %d\n"
            "Filter:      %s\n"
            "Resolving    %s ... ",
            gamestr,
            enctype,
            filter,
            ms);
    }

    peer.sin_addr.s_addr = msip = resolv(ms);
    peer.sin_port        = htons(msport);
    peer.sin_family      = AF_INET;

#ifdef SQL
    if(sql) {
        if(megaquery < 0) {
            fprintf(stderr, "\nError: you must specify both -Q and -S for using the SQL functions\n\n");
            exit(1);
        }
        gssql_init();
    }
#endif

    buff = malloc(BUFFSZ + 1);
    if(!buff) std_err();
    dynsz = BUFFSZ;

    if(!quiet) {
        fprintf(fdout,
            "%s:%hu\n",
            inet_ntoa(peer.sin_addr),
            msport);
    }

    if(hbmethod == 2) {
        printf("- the heartbeat for port %hu will be sent each %d minutes\n",
            HBMINUTES / 60,
            heartbeat_port);

        for(;;) {
            udpspr(0, &peer, GSHB1, heartbeat_port, gamestr);

            for(i = HBMINUTES; i; i--) {
                printf(" %3d\r", i);
                sleep(ONESEC);
            }
        }
        return(0);

    } else if(hbmethod == 1) {
        printf("- the game port %hu will be temporary bound each %d minutes, watch the details:\n",
            HBMINUTES / 60,
            heartbeat_port);

        peerl.sin_addr.s_addr = htonl(INADDR_ANY);
        peerl.sin_port        = htons(heartbeat_port);
        peerl.sin_family      = AF_INET;
        psz                   = sizeof(peer);

        for(;;) {
            sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
            if(sd < 0) std_err();

            printf("- binding: ");
            if(setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
              < 0) std_err();
            if(bind(sd, (struct sockaddr *)&peerl, sizeof(peerl))
              < 0) std_err();

            udpspr(sd, &peer, GSHB1, heartbeat_port, gamestr);
            if(!quiet) fputc('.', stdout);

            for(;;) {
                if(timeout(sd, TIMEOUT) < 0) {
                    fprintf(stderr, "\n"
                        "Error: socket timeout, probably your firewall blocks the UDP packets to or from the master server\n"
                        "\n");
                    exit(1);
                }

                err = recvfrom(sd, buff, BUFFSZ, 0, (struct sockaddr *)&peer, &psz);
                if(err <= 0) continue;

                    // here we check if the source IP is the same or is in the same B
                    // subnet of the master server, this is needed
                if((peer.sin_addr.s_addr & 0xffff) == (msip & 0xffff)) break;
                if(!quiet) fputc('x', stdout);
            }
            if(!quiet) fputc('.', stdout);
            buff[err] = 0;

            sec = keyval(buff, "secure");
            if(sec) {   // old, no longer used by the master server
                gsseckey(validate, sec, msgamekey, 0);
                printf("- Validate: %s -> %s\n", sec, validate);
                free(sec);
                udpspr(sd, &peer, GSHB2a, validate);
            } else {
                udpspr(sd, &peer, GSHB2b, gamestr);
            }
            if(!quiet) fputc('.', stdout);

            close(sd);
            printf(" free\n");

            for(i = HBMINUTES; i; i--) {
                printf(" %3d\r", i);
                sleep(ONESEC);
            }

                // needed to avoid that the sockaddr_in structure
                // is overwritten with a different host and port
            peer.sin_addr.s_addr = msip;
            peer.sin_port        = htons(msport);
            peer.sin_family      = AF_INET;
        }
        return(0);
    }

    multigamename = gamestr;

get_list:

    multigamenamep = strchr(gamestr, ',');
    if(multigamenamep) *multigamenamep = 0;
    if(!quiet) fprintf(fdout, "Gamename:    %s\n", gamestr);

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    sec = recv_basic_secure(sd, buff, BUFFSZ);
    if(sec) {
        STRCPY(secure, sec);
        gsseckey(validate, sec, msgamekey, enctype);
        if(!quiet) fprintf(fdout, "Validate:    %s -> %s\n", secure, validate);
        free(sec);
    } else {
        if(!quiet) {
            fprintf(stderr,
                "Alert: This master server doesn't seem to support the \"secure\" Gamespy protocol\n"
                "       This is the received reply: %s\n"
                "       I try to send the query with an empty \"validate\" field\n"
                "\n", buff);
        }
        *validate = 0;
        *secure   = 0;
    }

    if(tcpspr(sd,
        PCK,
        msgamename,
        enctype,
        validate,
        gamestr,
        *filter ? "\\where\\" : "",
        filter)
      < 0) std_err();

    switch(outtype) {
        case 0: break;
        case 1:
        case 3: {
            spr(&fname, "%s.gsl", gamestr);
            } break;
        case 2:
        case 4: {
            spr(&fname, "%s",     FILEOUT);
            } break;
        case 5:
        case 6: {
            fdout = stdout;
            } break;
    }

    if(fname) {
        if(!quiet) printf("- output file: %s\n", fname);
        fdout = fopen(fname, "wb");
        if(!fdout) std_err();
        free(fname);
        fname = NULL;
    }

    if(!quiet) printf("Receiving:   ");
    len = 0;
    while((err = recv(sd, buff + len, dynsz - len, 0)) > 0) {
        if(!quiet) fputc('.', stdout);
        len += err;
        if(len >= dynsz) {
            dynsz += BUFFSZ;
            buff = realloc(buff, dynsz);
            if(!buff) std_err();
        }
    }
    close(sd);

    if(!quiet) printf(" %u bytes\n", len);

    if(outtype == 6) {
        printf("\n\n");
        show_dump(buff, len, fdout);
        fputc('\n', stdout);
//        goto gslist_exit;
        outtype = 0;
    }

    ipport = (ipport_t *)buff;
    if(!enctype && !strncmp(buff + len - 7, "\\final\\", 7)) {
        len -= 7;
        itsok = 1;

    } else if((enctype == 1) && len) {
        ipport = (ipport_t *)enctype1_decoder(
            secure,
            buff,
            &len);
        itsok = 1;

    } else if((enctype == 2) && len) {
        ipport = (ipport_t *)enctype2_decoder(
            msgamekey,
            buff,
            &len);
        itsok = 1;
    }
    ipbuffer = ipport;

    if(!quiet) printf("-----------------------\n");

    if(!itsok && !quiet) {
        if(!len) {
            fprintf(stderr, "\n"
                "Error: the gamename doesn't exist in the master server\n"
                "\n");
        } else {
            fprintf(stderr, "\n"
                "Alert: the master server has not accepted your query for the following error:\n"
                "       %.*s\n"
                "\n", len, buff);
        }
        len = 0;
    }

    servers = 0;
    while(len >= 6) {
        ipc = inet_ntoa(*(struct in_addr *)&ipport->ip);

        switch(outtype) {
            case 0: {
                fprintf(fdout, "%15s   %hu\n", ipc, ntohs(ipport->port));
                } break;
            case 5:
            case 1:
            case 2: {
                fprintf(fdout, "%s:%hu\n", ipc, ntohs(ipport->port));
                } break;
            case 3:
            case 4: {
                fwrite((void *)ipport, 6, 1, fdout);
                } break;
        }

        if(execstring) {
            execptr = tmpexec + execlen;
            if(execstring_ip && !execstring_port) {
                execptr += sprintf(execptr, "%s", ipc);
                strcpy(execptr, execstring_ip + 3);

            } else if(execstring_port && !execstring_ip) {
                execptr += sprintf(execptr, "%hu", ntohs(ipport->port));
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_ip < execstring_port) {
                execptr += sprintf(execptr, "%s", ipc);
                execptr += sprintf(execptr, "%s", execstring_ip + 3);
                execptr += sprintf(execptr, "%hu", ntohs(ipport->port));
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_port < execstring_ip) {
                execptr += sprintf(execptr, "%hu", ntohs(ipport->port));
                execptr += sprintf(execptr, "%s", execstring_port + 5);
                execptr += sprintf(execptr, "%s", ipc);
                strcpy(execptr, execstring_ip + 3);
            }

            printf("   Execute: \"%s\"\n", tmpexec);
            system(tmpexec);
        }

        servers++;
        ipport++;
        len -= 6;
    }

    if(!quiet && itsok) printf("\n%u servers found\n\n", servers);

    if(megaquery >= 0) {
        fprintf(stderr, "- querying servers:\n");
        mega_query_scan(megaquery, ipbuffer, servers, 3);   // 3 is the default timeout
#ifdef SQL
        if(sql) gssql_later();
#endif
    }

    fflush(fdout);
    if(outtype) fclose(fdout);
        // -o filename will be closed when the program is terminated

    if(multigamenamep) {
        *multigamenamep = ',';
        gamestr = multigamenamep + 1;
        goto get_list;
    } else {
        gamestr = multigamename;
    }

    if(iwannaloop) {
        for(i = 0; i < iwannaloop; i++) {
            sleep(ONESEC);
        }
        goto get_list;
    }

//gslist_exit:
    if(tmpexec) free(tmpexec);
    free(buff);
    return(0);
}



void prepare_query(int query, u_char *ip, u_char *port) {
    u_char  *p = NULL;

    if(!ip || (!(p = strchr(ip, ':')) && !port)) {
        fprintf(stderr, "\n"
            "Error: you must insert an host and a query port\n"
            "\n");
        exit(1);
    }
    if(p) {
        *p++ = 0;
    } else {
        p = port;
    }
    multi_query(query, resolv(ip), atoi(p));
    fputc('\n', stdout);
}



int gsfullup_aluigi(void) {
    FILE    *fd;
    u_char  buff[GSLISTSZ + 1],
            *p;

    cool_download(GSLISTCFG,  ALUIGIHOST, 80, "papers/" GSLISTCFG);

    fd = gslfopen(GSLISTCFG, "rb");
    if(!fd) std_err();
    while(fgets(buff, sizeof(buff), fd)) {
        if(strncmp(buff, GSLISTVER, sizeof(GSLISTVER) - 1)) continue;
        p = buff + sizeof(GSLISTVER) - 1;
        delimit(p);
        if(strcmp(p, VER)) {
            printf(
                "- a new version of Gslist (%s) has been released:\n"
                "\n"
                "  "GSLISTHOME"\n"
                "\n", p);
        }
        break;
    }
    fclose(fd);

    return(0);
}


