/*
    Copyright 2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif



/*
finds the value of key in the data buffer and return a new
string containing the value or NULL if nothing has been found
no modifications are made on the input data
*/
u_char *keyval(u_char *data, u_char *key) {
    int     len,
            nt   = 0,
            skip = 1;
    u_char  *p,
            *val;

    for(;;) {
        p = strchr(data, '\\');

        if(nt & 1) {
            // key
            if(p && !strnicmp(data, key, p - data)) {
                skip = 0;
            }
        } else {
            // value
            if(!skip) {
                if(!p) p = data + strlen(data);
                len = p - data;
                val = malloc(len + 1);
                if(!val) std_err();
                memcpy(val, data, len);
                val[len] = 0;
                return(val);
            }
        }

        if(!p) break;
        nt++;
        data = p + 1;
    }
    return(NULL);
}



u_char *recv_basic_secure(int sd, u_char *data, int max) {
    int     t,
            len = 0;
    u_char  *sec = NULL;

    while(len < max) {
        t = recv(sd, data + len, max - len, 0);
        if(t < 0) std_err();
        if(!t) break;
        len += t;
        data[len] = 0;
        sec = keyval(data, "secure");
        if(sec) break;
        if(!data[len - 1]) break;
    }

    return(sec);
}



int timeout(int sock, int sec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



in_addr_t resolv(char *host) {
    struct      hostent *hp;
    in_addr_t   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == htonl(INADDR_NONE)) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n\n", host);
            exit(1);
        } else host_ip = *(in_addr_t *)(hp->h_addr);
    }
    return(host_ip);
}



u_char *delimit(u_char *data) {
    while(*data && (*data != '\n') && (*data != '\r')) data++;
    *data = 0;
    return(data);
}



int mystrcpy(u_char *dst, u_char *src, int max) {
    u_char  *p;

    for(p = dst; *src && --max; p++, src++) *p = *src;
    *p = 0;
    return(p - dst);
}



    /* get the input data and clean it */
int data_cleaner(u_char *dst, u_char *src, int max) {
    u_char  c,
            *p;

    for(p = dst; (c = *src) && (max > 0); src++, max--) {
        if(c == '^') {                      // Quake 3 colors
            continue;
        }
        if(c == 0x1b) {                     // Unreal colors
            src += 2;
            max -= 2;
            continue;
        }
        if(c < ' ') {                       // other colors
            continue;
        }
        if((c >= 0x7f) && (c <= 0x9f)) {
            continue;
        }
        switch(c) {
            case '%':
            case '&':
            case '\"':
            case '\'':
            case '<':
            case '>':               break;  // html/SQL injection
            case '\\': *p++ = '/';  break;  // avoids the delimiter
            case 0xa1: *p++ = '!';  break;  // ''
            case 0xa2: *p++ = 'c';  break;  // ''
            case 0xa3: *p++ = 'E';  break;  // ''
            case 0xa4: *p++ = 'x';  break;  // ''
            case 0xa5: *p++ = 'y';  break;  // ''
            case 0xa6: *p++ = '|';  break;  // ''
            case 0xa7: *p++ = 'S';  break;  // ''
            case 0xa8:              break;  // ''
            case 0xa9: *p++ = '';  break;  // ''
            case 0xaa: *p++ = 'a';  break;  // ''
            case 0xab: *p++ = '';  break;  // ''
            case 0xac:              break;  // ''
            case 0xad:              break;  // ''
            case 0xae: *p++ = 'R';  break;  // ''
            case 0xaf:              break;  // ''
            case 0xb0:              break;  // ''
            case 0xb1: *p++ = '+';  break;  // ''
            case 0xb2: *p++ = '2';  break;  // ''
            case 0xb3: *p++ = '3';  break;  // ''
            case 0xb4:              break;  // ''
            case 0xb5: *p++ = 'u';  break;  // ''
            case 0xb6:              break;  // ''
            case 0xb7:              break;  // ''
            case 0xb8: *p++ = '.';  break;  // ''
            case 0xb9: *p++ = '1';  break;  // ''
            case 0xba: *p++ = '.';  break;  // ''
            case 0xbb: *p++ = '';  break;  // ''
            case 0xbc:              break;  // ''
            case 0xbd:              break;  // ''
            case 0xbe:              break;  // ''
            case 0xbf: *p++ = '?';  break;  // ''
            case 0xc0: *p++ = 'A';  break;  // ''
            case 0xc1: *p++ = 'A';  break;  // ''
            case 0xc2: *p++ = 'A';  break;  // ''
            case 0xc3: *p++ = 'A';  break;  // ''
            case 0xc4: *p++ = 'A';  break;  // ''
            case 0xc5: *p++ = 'A';  break;  // ''
            case 0xc6: *p++ = 'E';  break;  // ''
            case 0xc7: *p++ = 'C';  break;  // ''
            case 0xc8: *p++ = 'E';  break;  // ''
            case 0xc9: *p++ = 'E';  break;  // ''
            case 0xca: *p++ = 'E';  break;  // ''
            case 0xcb: *p++ = 'E';  break;  // ''
            case 0xcc: *p++ = 'I';  break;  // ''
            case 0xcd: *p++ = 'I';  break;  // ''
            case 0xce: *p++ = 'I';  break;  // ''
            case 0xcf: *p++ = 'I';  break;  // ''
            case 0xd0: *p++ = 'D';  break;  // ''
            case 0xd1: *p++ = 'N';  break;  // ''
            case 0xd2: *p++ = 'O';  break;  // ''
            case 0xd3: *p++ = 'O';  break;  // ''
            case 0xd4: *p++ = 'O';  break;  // ''
            case 0xd5: *p++ = 'O';  break;  // ''
            case 0xd6: *p++ = 'O';  break;  // ''
            case 0xd7: *p++ = 'x';  break;  // ''
            case 0xd8: *p++ = '0';  break;  // ''
            case 0xd9: *p++ = 'U';  break;  // ''
            case 0xda: *p++ = 'U';  break;  // ''
            case 0xdb: *p++ = 'U';  break;  // ''
            case 0xdc: *p++ = 'U';  break;  // ''
            case 0xdd: *p++ = 'Y';  break;  // ''
            case 0xde: *p++ = 'd';  break;  // ''
            case 0xdf: *p++ = 'B';  break;  // ''
            case 0xe0: *p++ = 'a';  break;  // ''
            case 0xe1: *p++ = 'a';  break;  // ''
            case 0xe2: *p++ = 'a';  break;  // ''
            case 0xe3: *p++ = 'a';  break;  // ''
            case 0xe4: *p++ = 'a';  break;  // ''
            case 0xe5: *p++ = 'a';  break;  // ''
            case 0xe6: *p++ = 'e';  break;  // ''
            case 0xe7: *p++ = 'c';  break;  // ''
            case 0xe8: *p++ = 'e';  break;  // ''
            case 0xe9: *p++ = 'e';  break;  // ''
            case 0xea: *p++ = 'e';  break;  // ''
            case 0xeb: *p++ = 'e';  break;  // ''
            case 0xec: *p++ = 'i';  break;  // ''
            case 0xed: *p++ = 'i';  break;  // ''
            case 0xee: *p++ = 'i';  break;  // ''
            case 0xef: *p++ = 'i';  break;  // ''
            case 0xf0: *p++ = 'o';  break;  // ''
            case 0xf1: *p++ = 'n';  break;  // ''
            case 0xf2: *p++ = 'o';  break;  // ''
            case 0xf3: *p++ = 'o';  break;  // ''
            case 0xf4: *p++ = 'o';  break;  // ''
            case 0xf5: *p++ = 'o';  break;  // ''
            case 0xf6: *p++ = 'o';  break;  // ''
            case 0xf7: *p++ = '+';  break;  // ''
            case 0xf8: *p++ = '0';  break;  // ''
            case 0xf9: *p++ = 'u';  break;  // ''
            case 0xfa: *p++ = 'u';  break;  // ''
            case 0xfb: *p++ = 'u';  break;  // ''
            case 0xfc: *p++ = 'u';  break;  // ''
            case 0xfd: *p++ = 'y';  break;  // ''
            case 0xfe: *p++ = 'b';  break;  // ''
            case 0xff: *p++ = 'y';  break;  // ''
            default:   *p++ = c;    break;
        }
    }
    *p = 0;

    return(p - dst);
}



    /* returns a new clean string */
u_char *mychrdup(u_char *src) {
    int     len,
            max;
    u_char  *dst;

    len = max = strlen(src);
    dst = malloc(max + 1);
    if(!dst) std_err();

    max = data_cleaner(dst, src, max);

        // probably useless free memory
    if(max < len) {
        dst = realloc(dst, max + 1);
        if(!dst) std_err();
    }

    return(dst);
}



int mysnprintf(u_char *buff, int len, u_char *fmt, ...) {
    va_list ap;
    int     ret;

    va_start(ap, fmt);
    ret = vsnprintf(buff, len, fmt, ap);
    va_end(ap);

    if((ret < 0) || (ret >= len)) {
        ret = len;
        buff[len] = 0;
    }
    return(ret);
}



    /*
    the core of the local config files
    it works just like the normal fopen()
    if you use a NULL mode will be returned the (allocated)
    full path of the file
    */
void *gslfopen(const char *path, const char *mode) {
#ifdef WIN32
    HKEY        key;
    char        home[GSMAXPATH + 1];
    int         len;
#else
    const char  *home;
#endif
    char        retpath[GSMAXPATH + 64 + 1];

    if(!*gslist_path) {
#ifdef WIN32
        len = GSMAXPATH;
        if(!RegOpenKeyEx(HKEY_CURRENT_USER, APPDATA, 0, KEY_READ, &key)) {
            if(!RegQueryValueEx(key, "AppData", NULL, NULL, home, (void *)&len)) {
                mysnprintf(gslist_path, sizeof(gslist_path), "%s\\gslist\\", home);
                mkdir(gslist_path);
            }
            RegCloseKey(key);
        }
        if(!*gslist_path) strcpy(gslist_path, ".\\");
#else
        home = getenv("HOME");
        if(!home) {
            fputs("\n"
                "Error: impossible to know your $HOME or Application Data directory where\n"
                "       reading/writing the Gslist configuration files.\n"
                "       Modify the source code of the program or report the problem to me.\n"
                "\n", stderr);
            exit(1);
        }
        mysnprintf(gslist_path, sizeof(gslist_path), "%s/.gslist/", home);
        mkdir(gslist_path, 0755);
#endif
    }

//    mysnprintf(retpath, sizeof(retpath), "%s%s", gslist_path, path);
    sprintf(retpath, "%s%s", gslist_path, path);

    if(!mode) return(strdup(retpath));
    return(fopen(retpath, mode));
}



u_char *replace(u_char *buff, u_char *from, u_char *to) {
    u_char  *p;
    int     fromlen,
            tolen;

    p = (u_char *)stristr(buff, from);
    if(!p) return(buff);

    fromlen = strlen(from);
    tolen   = strlen(to);
    memmove(p + tolen, p + fromlen, strlen(p) + 1);
    memcpy(p, to, tolen);
    return(buff);
}


int vspr(u_char **buff, u_char *fmt, va_list ap) {
    int     len,
            mlen;
    u_char  *ret;

    mlen = strlen(fmt) + 128;

    for(;;) {
        ret = malloc(mlen);
        if(!ret) return(0);     // return(-1);
        len = vsnprintf(ret, mlen, fmt, ap);
        if((len >= 0) && (len < mlen)) break;
        if(len < 0) {           // Windows style
            mlen += 128;
        } else {                // POSIX style
            mlen = len + 1;
        }
        free(ret);
    }

    *buff = ret;
    return(len);
}



int spr(u_char **buff, u_char *fmt, ...) {
    va_list ap;
    int     len;

    va_start(ap, fmt);
    len = vspr(buff, fmt, ap);
    va_end(ap);

    return(len);
}



int udpspr(int sd, struct sockaddr_in *peer, u_char *fmt, ...) {
    va_list ap;
    int     len,
            new = 0;
    u_char  *buff;

    va_start(ap, fmt);
    len = vspr(&buff, fmt, ap);
    va_end(ap);

    if(!sd) {
        new = 1;
        sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sd < 0) std_err();
    }

    len = sendto(sd, buff, len, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in));
    if(new) close(sd);
    free(buff);
    return(len);
}



int tcpspr(int sd, u_char *fmt, ...) {
    va_list ap;
    int     len;
    u_char  *buff;

    va_start(ap, fmt);
    len = vspr(&buff, fmt, ap);
    va_end(ap);

    len = send(sd, buff, len, 0);
    free(buff);
    return(len);
}



    /* returns -1 if error, 1 if only par, 2 if par and val, 0 if [] */
int myreadini(FILE *fd, u_char *line, int linesz, u_char **par, u_char **val) {
    u_char  *p,
            *l;

    *par = NULL;
    *val = NULL;

    if(!fgets(line, linesz, fd)) return(-1);

    delimit(line);

    while(*line && (*line <= ' ') && (*line != '=')) line++;    // first spaces
    *par = line;

    while(*line && (*line > ' ')  && (*line != '=')) line++;    // delimit par
    *line++ = 0;

    while(*line && (*line <= ' ') && (*line != '=')) line++;    // remove spaces after par
    *val = line;

    line += strlen(line);                                       // remove last spaces
    while(*line && (*line <= ' ')) line--;
    *(line + 1) = 0;

    p = *par;
    if(*p == '[') {
        p++;
        l = strchr(p, ']');
        if(l) *l = 0;
        if(strlen(p) > CNAMELEN) p[CNAMELEN] = 0;
        *par = p;
        return(0);
    }
    if(!(*val)[0]) return(1);
    return(2);
}

